#include <stdint.h>
#include <util/delay.h>

#include "BusProtectionInterface.h"
#include "../Automaton/Automaton.h"
#include "../UART0.h"

uint8_t *BusProtectionOutputBuffer;
uint8_t *BusProtectionInputBuffer;

uint8_t ReceiveIndex;
uint8_t TmpBuffer[6];

void BusProtectionInterfaceInit(void)
{
	BusProtectionOutputBuffer = ActuatorPointer;
	BusProtectionInputBuffer = SensorPointer;
	ReceiveIndex = 0;
}

void BusProtectionSendData(void)
{
	UART0_put_char(0xFF);
	_delay_ms(1);
	for (uint8_t i=0; i<6;i++)
	{
		if(BusProtectionOutputBuffer[i] >= 0x80)
		{
			UART0_put_char(0x80);
			
			_delay_ms(1);
			UART0_put_char(BusProtectionOutputBuffer[i]-0x80);
			
			_delay_ms(1);
		}else
		{
			UART0_put_char(0);
			
			_delay_ms(1);
			UART0_put_char(BusProtectionOutputBuffer[i]);
			
			_delay_ms(1);
		}
	}
	
	UART0_put_char(0xFE);
	_delay_ms(100);
}

void BusProtectionNewData(uint8_t Data)
{
	switch(Data)
	{
		case 0xFF:
			//start data transfer
			ReceiveIndex=0;
			break;
		case 0xFE:
			//data transfer ended
			for (uint8_t i=0; i<6;i++)
			{
				BusProtectionInputBuffer[i] = TmpBuffer[i];
			}
			ReceiveIndex=0;
			break;
		default:
			if(ReceiveIndex%2 == 1)
			{
				TmpBuffer[ReceiveIndex/2]+= Data;
			}else
				TmpBuffer[ReceiveIndex/2] = Data;
			ReceiveIndex++;
			break;
	}
}


void BusProtectionSetAktor(uint8_t AktorBit)
{
	uint8_t ByteIndex = AktorBit / 8;
	uint8_t ActuatorBitInByte = AktorBit % 8;
	
	BusProtectionOutputBuffer[ByteIndex] |=  (1<<ActuatorBitInByte);
}

void BusProtectionResetAktor(uint8_t AktorBit)
{
	uint8_t ByteIndex = AktorBit / 8;
	uint8_t ActuatorBitInByte = AktorBit % 8;
	
	BusProtectionOutputBuffer[ByteIndex] &=  ~(1<<ActuatorBitInByte);
}


void BusProtectionSetSignal(uint8_t StartBit,uint8_t BitCount, uint64_t Value)
{
	uint8_t ByteIndex = StartBit / 8;
	uint8_t StartBitInByte = StartBit % 8;
	uint8_t ShiftPosition = 0;
	
	BusProtectionOutputBuffer[ByteIndex] &=  ~(0xFF << StartBitInByte);
	BusProtectionOutputBuffer[ByteIndex] |=  ((Value<<StartBitInByte) & (0xFF << StartBitInByte));
	
	if((StartBitInByte+BitCount) > 8)
	{
		ByteIndex++;
		BitCount -= (8-StartBitInByte);
		ShiftPosition += (8-StartBitInByte);
		while (BitCount>8)
		{
			BusProtectionOutputBuffer[ByteIndex] = (Value >> ShiftPosition) & 0xFF;
			ShiftPosition+=8;
			BitCount-= 8;
			ByteIndex++;
		}
		
		BusProtectionOutputBuffer[ByteIndex] &=  ~(0xFF >> (8-BitCount));
		BusProtectionOutputBuffer[ByteIndex] |=  ((Value>>ShiftPosition) & (0xFF >> (8-BitCount)));
	}
}

uint64_t BusProtectionGetSignal(uint8_t StartBit, uint8_t BitCount)
{
	uint64_t Result = 0;
	uint8_t ShiftPosition = 0;
	uint8_t ByteIndex = StartBit / 8;
	uint8_t StartBitInByte = StartBit % 8;
	if((StartBitInByte+BitCount) <= 8)
	{
		Result = (BusProtectionInputBuffer[ByteIndex] >> StartBitInByte) & ~(0xFF << BitCount);
	}else
	{
		Result = (BusProtectionInputBuffer[ByteIndex] >> StartBitInByte);
		BitCount -= (8 - StartBitInByte);
		ShiftPosition += (8 - StartBitInByte);
		ByteIndex++;
		while (BitCount > 8)
		{
			Result |= (BusProtectionInputBuffer[ByteIndex] << ShiftPosition);
			ByteIndex++;
			ShiftPosition += 8;
			BitCount -= 8;
		}
		Result |= (BusProtectionInputBuffer[ByteIndex] & ~(0xFF << BitCount))<< ShiftPosition;
	}
	return Result;
}
